<?php
/**
 * Widget API: Idblog_Mostview_Wgt class
 *
 * Author: Gian MR - http://www.gianmr.com
 *
 * @package Idblog Core
 * @subpackage Widgets
 * @since 1.0.0
 */

/* Exit if accessed directly */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Add the Most view widget.
 *
 * @since 1.0.0
 *
 * @see WP_Widget
 */
class Idblog_Mostview_Wgt extends WP_Widget {
	/**
	 * Sets up a Most view Posts widget instance.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function __construct() {
		$widget_ops = array(
			'classname'   => 'idblog-form',
			'description' => __( 'Most view posts with thumbnails widget.', 'idblog-core' ),
		);
		parent::__construct( 'idblog-mostview', __( 'Most view Posts (Idblog)', 'idblog-core' ), $widget_ops );

		// add action for admin_register_scripts.
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_register_scripts' ) );
		add_action( 'admin_footer-widgets.php', array( $this, 'admin_print_scripts' ), 9999 );
	}

	/**
	 * Enqueue scripts.
	 *
	 * @since 1.0
	 *
	 * @param string $hook_suffix Hook Suffix.
	 */
	public function admin_register_scripts( $hook_suffix ) {
		if ( 'widgets.php' !== $hook_suffix ) {
			return;
		}

		wp_enqueue_style( 'wp-color-picker' );
		wp_enqueue_script( 'wp-color-picker' );
		wp_enqueue_script( 'underscore' );
	}

	/**
	 * Print scripts.
	 *
	 * @since 1.0
	 */
	public function admin_print_scripts() {
		?>
		<script>
			( function( $ ){
				function initColorPicker( widget ) {
					widget.find( '.color-picker' ).wpColorPicker( {
						change: _.throttle( function() { // For Customizer
							$(this).trigger( 'change' );
						}, 3000 )
					});
				}

				function onFormUpdate( event, widget ) {
					initColorPicker( widget );
				}

				$( document ).on( 'widget-added widget-updated', onFormUpdate );

				$( document ).ready( function() {
					$( '#widgets-right .widget:has(.color-picker)' ).each( function () {
						initColorPicker( $( this ) );
					} );
				} );
			}( jQuery ) );
		</script>
		<?php
	}

	/**
	 * Outputs the content for Mailchimp Form.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $args     Display arguments including 'before_title', 'after_title',
	 *                        'before_widget', and 'after_widget'.
	 * @param array $instance Settings for Mailchimp Form.
	 */
	public function widget( $args, $instance ) {

		// Title.
		$title = apply_filters( 'widget_title', empty( $instance['title'] ) ? '' : $instance['title'], $instance, $this->id_base );

		echo $args['before_widget']; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		if ( $title ) {
			echo $args['before_title'] . $title . $args['after_title']; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
		// Base Id Widget.
		$idblg_widget_id = $this->id_base . '-' . $this->number;
		// Category ID.
		$idblg_category_ids = ( ! empty( $instance['idblg_category_ids'] ) ) ? array_map( 'absint', $instance['idblg_category_ids'] ) : array( 0 );
		// Excerpt Length.
		$idblg_number_posts = ( ! empty( $instance['idblg_number_posts'] ) ) ? absint( $instance['idblg_number_posts'] ) : absint( 10 );
		// Excerpt Length.
		$idblg_excerpt_length = ( ! empty( $instance['idblg_excerpt_length'] ) ) ? absint( $instance['idblg_excerpt_length'] ) : absint( apply_filters( 'idblg_excerpt_length_filter', 55 ) );
		// Excerpt More Text.
		$idblg_excerpt_more = empty( $instance['idblg_excerpt_more'] ) ? '[&hellip;]' : wp_strip_all_tags( $instance['idblg_excerpt_more'] );
		// Title Length.
		$idblg_title_length = ( ! empty( $instance['idblg_title_length'] ) ) ? absint( $instance['idblg_title_length'] ) : absint( 40 );
		// Popular by date.
		$idblg_popular_date = ( isset( $instance['idblg_popular_date'] ) ) ? esc_attr( $instance['idblg_popular_date'] ) : esc_attr( 'alltime' );
		// Hide current post.
		$idblg_hide_current_post    = ( isset( $instance['idblg_hide_current_post'] ) ) ? (bool) $instance['idblg_hide_current_post'] : false;
		$idblg_show_author          = ( isset( $instance['idblg_show_author'] ) ) ? (bool) $instance['idblg_show_author'] : false;
		$idblg_show_categories      = ( isset( $instance['idblg_show_categories'] ) ) ? (bool) $instance['idblg_show_categories'] : false;
		$idblg_show_comments_number = ( isset( $instance['idblg_show_comments_number'] ) ) ? (bool) $instance['idblg_show_comments_number'] : false;
		$idblg_show_view            = ( isset( $instance['idblg_show_view'] ) ) ? (bool) $instance['idblg_show_view'] : false;
		$idblg_show_excerpt         = ( isset( $instance['idblg_show_excerpt'] ) ) ? (bool) $instance['idblg_show_excerpt'] : false;
		$idblg_show_thumb           = ( isset( $instance['idblg_show_thumb'] ) ) ? (bool) $instance['idblg_show_thumb'] : false;

		// Style.
		$bgcolor    = ( ! empty( $instance['bgcolor'] ) ) ? wp_strip_all_tags( $instance['bgcolor'] ) : '';
		$color_text = ( ! empty( $instance['color_text'] ) ) ? wp_strip_all_tags( $instance['color_text'] ) : '';
		$color_link = ( ! empty( $instance['color_link'] ) ) ? wp_strip_all_tags( $instance['color_link'] ) : '';
		$color_meta = ( ! empty( $instance['color_meta'] ) ) ? wp_strip_all_tags( $instance['color_meta'] ) : '';

		// Banner.
		$idblg_bannerinfeed = ( ! empty( $instance['idblg_bannerinfeed'] ) ) ? $instance['idblg_bannerinfeed'] : '';

		// if 'all categories' was selected ignore other selections of categories.
		if ( in_array( 0, $idblg_category_ids, true ) ) {
			$idblg_category_ids = array( 0 );
		}

		// standard params.
		$query_args = array(
			'posts_per_page'         => $idblg_number_posts,
			'no_found_rows'          => true,
			'post_status'            => 'publish',
			// make it fast withour update term cache and cache results
			// https://thomasgriffin.io/optimize-wordpress-queries/.
			'update_post_term_cache' => false,
			'update_post_meta_cache' => false,
		);

		$query_args['ignore_sticky_posts'] = true;
		$query_args['meta_key']            = 'views';
		$query_args['orderby']             = 'meta_value_num';

		// set order of posts in widget.
		$query_args['order'] = 'DESC';

		if ( 'weekly' === $idblg_popular_date ) {
			// Get posts last week.
			$query_args['date_query'] = array(
				array(
					'after' => '1 week ago',
				),
			);
		} elseif ( 'mountly' === $idblg_popular_date ) {
			// Get posts last mount.
			$query_args['date_query'] = array(
				array(
					'after' => '1 month ago',
				),
			);
		} elseif ( 'secondmountly' === $idblg_popular_date ) {
			// Get posts last mount.
			$query_args['date_query'] = array(
				array(
					'after' => '2 months ago',
				),
			);
		} elseif ( 'yearly' === $idblg_popular_date ) {
			// Get posts last mount.
			$query_args['date_query'] = array(
				array(
					'after' => '1 year ago',
				),
			);
		}

		// add categories param only if 'all categories' was not selected.
		if ( ! in_array( 0, $idblg_category_ids, true ) ) {
			$query_args['category__in'] = $idblg_category_ids;
		}

		// exclude current displayed post.
		if ( $idblg_hide_current_post ) {
			global $post;
			if ( isset( $post->ID ) && is_singular() ) {
				$query_args['post__not_in'] = array( $post->ID );
			}
		}

		// run the query: get the latest posts.
		$rp = new WP_Query( apply_filters( 'idblg_rp_widget_posts_args', $query_args ) );
		if ( $bgcolor ) {
			$color = ' style="padding:20px;background-color:' . esc_html( $bgcolor ) . '"';
		} else {
			$color = '';
		}
		if ( $color_link ) {
			$colorlink = ' style="color:' . esc_html( $color_link ) . '"';
		} else {
			$colorlink = '';
		}
		if ( $color_meta ) {
			$colormeta = ' style="color:' . esc_html( $color_meta ) . '"';
		} else {
			$colormeta = '';
		}
		if ( $color_text ) {
			$colortext = ' style="color:' . esc_html( $color_text ) . '"';
		} else {
			$colortext = '';
		}
		?>

			<div class="idblog-rp-widget"<?php echo $color; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>>
				<div class="idblog-rp">
					<ul>
						<?php
						while ( $rp->have_posts() ) :
							$rp->the_post();
							echo '<li id="listpost"';
							if ( is_sticky() ) {
								echo ' class="idblog-rp-sticky"';
							}
							echo '>';
							?>
							<div class="idblog-rp-link clearfix">
								<a href="<?php the_permalink(); ?>"<?php echo $colorlink;  // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?> itemprop="url" title="<?php the_title_attribute( array( 'before' => __( 'Permalink to: ', 'idblog-core' ), 'after' => '' ) ); ?>">
									<?php
									if ( $idblg_show_thumb ) :
										// look for featured image.
										if ( has_post_thumbnail() ) :
											the_post_thumbnail( 'thumbnail', array( 'itemprop' => 'image' ) );
										endif; // has_post_thumbnail.
									endif; // show_thumb.
									?>

									<span class="idblog-rp-title">
										<?php
										if ( $post_title = $this->get_the_trimmed_post_title( $idblg_title_length ) ) {
											echo esc_html( $post_title );
										} else {
											the_title();
										}
										?>
									</span>
								</a>
								<?php
								if ( $idblg_show_view ) :
									?>
									<div class="idblog-rp-meta idblog-rp-date"<?php echo $colormeta; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>>
										<?php
										if ( function_exists( 'the_views' ) ) {
											the_views();
										}
										?>
									</div>
								<?php endif; ?>
							</div>
							<?php
							if ( $idblg_show_excerpt ) :
								?>
								<div class="idblog-rp-excerpt"<?php echo $colortext; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>>
									<?php echo $this->get_the_trimmed_excerpt( $idblg_excerpt_length, $idblg_excerpt_more ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
								</div>
								<?php
							endif;
							if ( $idblg_show_author ) :
								?>
								<div class="idblog-rp-meta idblog-rp-author"<?php echo $colormeta; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>>
									<?php echo esc_html__( 'By', 'idblog-core' ); ?> <?php the_author(); ?>
								</div>
								<?php
							endif;

							if ( $idblg_show_categories ) :
								?>
								<div class="idblog-rp-meta idblog-rp-categories"<?php echo $colormeta; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>>
									<?php echo $this->get_the_categories( $rp->post->ID ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
								</div>
								<?php
							endif;

							if ( $idblg_show_comments_number ) :
								?>
								<div class="idblog-rp-meta idblog-rp-comments"<?php echo $colormeta; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>>
									<?php echo get_comments_number_text(); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
								</div>
								<?php
							endif;
							echo '</li>';
							if ( $idblg_bannerinfeed ) {
								if ( 1 === $rp->current_post ) {
									echo '<li class="banner">';
										echo do_shortcode( htmlspecialchars_decode( $idblg_bannerinfeed ) );
									echo '</li>';

								}
							}
						endwhile;
						?>
					</ul>
				</div>
			</div>

		<?php
		echo $args['after_widget']; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}

	/**
	 * Handles updating settings for the current Mailchimp widget instance.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $new_instance New settings for this instance as input by the user via
	 *                            Idblog_Mailchimp_form::form().
	 * @param array $old_instance Old settings for this instance.
	 * @return array Updated settings to save.
	 */
	public function update( $new_instance, $old_instance ) {
		$instance     = $old_instance;
		$new_instance = wp_parse_args(
			(array) $new_instance,
			array(
				'title'                      => '',
				'idblg_category_ids'         => array( 0 ),
				'idblg_number_posts'         => 10,
				'idblg_excerpt_length'       => 55,
				'idblg_excerpt_more'         => '[&hellip;]',
				'idblg_title_length'         => 40,
				'idblg_hide_current_post'    => false,
				'idblg_show_author'          => false,
				'idblg_show_categories'      => false,
				'idblg_show_comments_number' => false,
				'idblg_popular_date'         => 'alltime',
				'idblg_show_view'            => false,
				'idblg_show_excerpt'         => false,
				'idblg_show_thumb'           => false,
				'idblg_bannerinfeed'         => '',
				'bgcolor'                    => '',
				'color_text'                 => '',
				'color_link'                 => '',
				'color_meta'                 => '',
			)
		);
		// Title.
		$instance['title'] = sanitize_text_field( $new_instance['title'] );
		// Category IDs.
		$instance['idblg_category_ids'] = array_map( 'absint', $new_instance['idblg_category_ids'] );
		// Number posts.
		$instance['idblg_number_posts'] = absint( $new_instance['idblg_number_posts'] );
		// Excerpt Length.
		$instance['idblg_excerpt_length'] = absint( $new_instance['idblg_excerpt_length'] );
		// Excerpt More Text.
		$instance['idblg_excerpt_more'] = wp_strip_all_tags( $new_instance['idblg_excerpt_more'] );
		// Title Length.
		$instance['idblg_title_length'] = absint( $new_instance['idblg_title_length'] );
		// Popular range.
		$instance['idblg_popular_date'] = esc_attr( $new_instance['idblg_popular_date'] );
		// Hide current post.
		$instance['idblg_hide_current_post'] = (bool) $new_instance['idblg_hide_current_post'];
		// Show element.
		$instance['idblg_show_author']          = (bool) $new_instance['idblg_show_author'];
		$instance['idblg_show_categories']      = (bool) $new_instance['idblg_show_categories'];
		$instance['idblg_show_comments_number'] = (bool) $new_instance['idblg_show_comments_number'];
		$instance['idblg_show_view']            = (bool) $new_instance['idblg_show_view'];
		$instance['idblg_show_excerpt']         = (bool) $new_instance['idblg_show_excerpt'];
		$instance['idblg_show_thumb']           = (bool) $new_instance['idblg_show_thumb'];
		// Style.
		$instance['bgcolor']    = wp_strip_all_tags( $new_instance['bgcolor'] );
		$instance['color_text'] = wp_strip_all_tags( $new_instance['color_text'] );
		$instance['color_link'] = wp_strip_all_tags( $new_instance['color_link'] );
		$instance['color_meta'] = wp_strip_all_tags( $new_instance['color_meta'] );
		// In feed banner.
		$instance['idblg_bannerinfeed'] = $new_instance['idblg_bannerinfeed'];

		// if 'all categories' was selected ignore other selections of categories.
		if ( in_array( 0, $instance['idblg_category_ids'], true ) ) {
			$instance['idblg_category_ids'] = array( 0 );
		}

		return $instance;
	}

	/**
	 * Outputs the settings form for the Mailchimp widget.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $instance Current settings.
	 */
	public function form( $instance ) {
		$instance = wp_parse_args(
			(array) $instance,
			array(
				'title'                      => 'Most View Post',
				'idblg_category_ids'         => array( 0 ),
				'idblg_number_posts'         => 10,
				'idblg_excerpt_length'       => 55,
				'idblg_excerpt_more'         => '[&hellip;]',
				'idblg_title_length'         => 40,
				'idblg_hide_current_post'    => false,
				'idblg_show_author'          => false,
				'idblg_show_categories'      => false,
				'idblg_show_comments_number' => false,
				'idblg_popular_date'         => 'alltime',
				'idblg_show_view'            => true,
				'idblg_show_excerpt'         => false,
				'idblg_show_thumb'           => true,
				'idblg_bannerinfeed'         => '',
				'bgcolor'                    => '',
				'color_text'                 => '',
				'color_link'                 => '',
				'color_meta'                 => '',
			)
		);
		// Title.
		$title = sanitize_text_field( $instance['title'] );
		// Category ID.
		$idblg_category_ids = array_map( 'absint', $instance['idblg_category_ids'] );
		// Number posts.
		$idblg_number_posts = absint( $instance['idblg_number_posts'] );
		// Excerpt Length.
		$idblg_excerpt_length = absint( $instance['idblg_excerpt_length'] );
		// Excerpt More Text.
		$idblg_excerpt_more = wp_strip_all_tags( $instance['idblg_excerpt_more'] );
		// Title Length.
		$idblg_title_length = absint( $instance['idblg_title_length'] );
		// Popular range.
		$idblg_popular_date = esc_attr( $instance['idblg_popular_date'] );
		// Hide current post.
		$idblg_hide_current_post = (bool) $instance['idblg_hide_current_post'];
		// Show element.
		$idblg_show_author          = (bool) $instance['idblg_show_author'];
		$idblg_show_categories      = (bool) $instance['idblg_show_categories'];
		$idblg_show_comments_number = (bool) $instance['idblg_show_comments_number'];
		$idblg_show_view            = (bool) $instance['idblg_show_view'];
		$idblg_show_excerpt         = (bool) $instance['idblg_show_excerpt'];
		$idblg_show_thumb           = (bool) $instance['idblg_show_thumb'];
		// Style.
		$bgcolor    = wp_strip_all_tags( $instance['bgcolor'] );
		$color_text = wp_strip_all_tags( $instance['color_text'] );
		$color_link = wp_strip_all_tags( $instance['color_link'] );
		$color_meta = wp_strip_all_tags( $instance['color_meta'] );

		// In feed banner.
		$idblg_bannerinfeed = $instance['idblg_bannerinfeed'];

		// get categories.
		$categories     = get_categories(
			array(
				'hide_empty'   => 0,
				'hierarchical' => 1,
			)
		);
		$number_of_cats = count( $categories );

		// get size (number of rows to display) of selection box: not more than 10.
		$number_of_rows = ( 10 > $number_of_cats ) ? $number_of_cats + 1 : 10;

		// if 'all categories' was selected ignore other selections of categories.
		if ( in_array( 0, $idblg_category_ids, true ) ) {
			$idblg_category_ids = array( 0 );
		}

		// start selection box.
		$selection_category  = sprintf(
			'<select name="%s[]" id="%s" class="cat-select widefat" multiple size="%d">',
			$this->get_field_name( 'idblg_category_ids' ),
			$this->get_field_id( 'idblg_category_ids' ),
			$number_of_rows
		);
		$selection_category .= "\n";

		// make selection box entries.
		$cat_list = array();
		if ( 0 < $number_of_cats ) {

			// make a hierarchical list of categories.
			while ( $categories ) {
				// go on with the first element in the categories list:.
				// if there is no parent.
				if ( '0' == $categories[0]->parent ) {
					// get and remove it from the categories list.
					$current_entry = array_shift( $categories );
					// append the current entry to the new list.
					$cat_list[] = array(
						'id'    => absint( $current_entry->term_id ),
						'name'  => esc_html( $current_entry->name ),
						'depth' => 0,
					);
					// go on looping.
					continue;
				}
				// if there is a parent:
				// try to find parent in new list and get its array index.
				$parent_index = $this->get_cat_parent_index( $cat_list, $categories[0]->parent );
				// if parent is not yet in the new list: try to find the parent later in the loop.
				if ( false === $parent_index ) {
					// get and remove current entry from the categories list.
					$current_entry = array_shift( $categories );
					// append it at the end of the categories list.
					$categories[] = $current_entry;
					// go on looping.
					continue;
				}
				// if there is a parent and parent is in new list:
				// set depth of current item: +1 of parent's depth.
				$depth = $cat_list[ $parent_index ]['depth'] + 1;
				// set new index as next to parent index.
				$new_index = $parent_index + 1;
				// find the correct index where to insert the current item.
				foreach ( $cat_list as $entry ) {
					// if there are items with same or higher depth than current item.
					if ( $depth <= $entry['depth'] ) {
						// increase new index.
						$new_index = $new_index + 1;
						// go on looping in foreach().
						continue;
					}
					// if the correct index is found:
					// get current entry and remove it from the categories list.
					$current_entry = array_shift( $categories );
					// insert current item into the new list at correct index.
					$end_array  = array_splice( $cat_list, $new_index ); // $cat_list is changed, too.
					$cat_list[] = array(
						'id'    => absint( $current_entry->term_id ),
						'name'  => esc_html( $current_entry->name ),
						'depth' => $depth,
					);
					$cat_list   = array_merge( $cat_list, $end_array );
					// quit foreach(), go on while-looping.
					break;
				}
			}

			// make HTML of selection box.
			$selected            = ( in_array( 0, $idblg_category_ids, true ) ) ? ' selected="selected"' : '';
			$selection_category .= "\t";
			$selection_category .= '<option value="0"' . $selected . '>' . __( 'All Categories', 'idblog-core' ) . '</option>';
			$selection_category .= "\n";

			foreach ( $cat_list as $category ) {
				$cat_name            = apply_filters( 'idblog_list_cats', $category['name'], $category );
				$pad                 = ( 0 < $category['depth'] ) ? str_repeat( '&ndash;&nbsp;', $category['depth'] ) : '';
				$selection_category .= "\t";
				$selection_category .= '<option value="' . $category['id'] . '"';
				$selection_category .= ( in_array( $category['id'], $idblg_category_ids, true ) ) ? ' selected="selected"' : '';
				$selection_category .= '>' . $pad . $cat_name . '</option>';
				$selection_category .= "\n";
			}
		}

		// close selection box.
		$selection_category .= "</select>\n";

		?>

		<p>
			<label for="<?php echo esc_html( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title:', 'idblog-core' ); ?></label>
			<input class="widefat" id="<?php echo esc_html( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
		</p>
		<p>
			<label for="<?php echo esc_html( $this->get_field_id( 'idblg_category_ids' ) ); ?>"><?php esc_html_e( 'Selected categories', 'idblog-core' ); ?></label>
			<?php echo $selection_category; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
			<br />
			<small><?php esc_html_e( 'Click on the categories with pressed CTRL key to select multiple categories. If All Categories was selected then other selections will be ignored.', 'idblog-core' ); ?></small>
		</p>
		<p>
			<label for="<?php echo esc_html( $this->get_field_id( 'idblg_number_posts' ) ); ?>"><?php esc_html_e( 'Number post', 'idblog-core' ); ?></label>
			<input class="widefat" id="<?php echo esc_html( $this->get_field_id( 'idblg_number_posts' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'idblg_number_posts' ) ); ?>" type="number" value="<?php echo esc_attr( $idblg_number_posts ); ?>" />
		</p>
		<p>
			<label for="<?php echo esc_html( $this->get_field_id( 'idblg_excerpt_length' ) ); ?>"><?php esc_html_e( 'Maximum length of excerpt', 'idblog-core' ); ?></label>
			<input class="widefat" id="<?php echo esc_html( $this->get_field_id( 'idblg_excerpt_length' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'idblg_excerpt_length' ) ); ?>" type="number" value="<?php echo esc_attr( $idblg_excerpt_length ); ?>" />
		</p>
		<p>
			<label for="<?php echo esc_html( $this->get_field_id( 'idblg_popular_date' ) ); ?>"><?php esc_html_e( 'Popular range:', 'idblog-core' ); ?></label>
			<select class="widefat" id="<?php echo esc_html( $this->get_field_id( 'idblg_popular_date' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'idblg_popular_date' ) ); ?>">
				<option value="alltime" <?php selected( $instance['idblg_popular_date'], 'alltime' ); ?>><?php esc_html_e( 'Alltime', 'idblog-core' ); ?></option>
				<option value="yearly" <?php selected( $instance['idblg_popular_date'], 'yearly' ); ?>><?php esc_html_e( '1 Year', 'idblog-core' ); ?></option>
				<option value="secondmountly" <?php selected( $instance['idblg_popular_date'], 'secondmountly' ); ?>><?php esc_html_e( '2 Mounts', 'idblog-core' ); ?></option>
				<option value="mountly" <?php selected( $instance['idblg_popular_date'], 'mountly' ); ?>><?php esc_html_e( '1 Mount', 'idblog-core' ); ?></option>
				<option value="weekly" <?php selected( $instance['idblg_popular_date'], 'weekly' ); ?>><?php esc_html_e( '7 Days', 'idblog-core' ); ?></option>
			</select>
			<br/>
			<small><?php esc_html_e( 'Select popular by most view.', 'idblog-core' ); ?></small>
		</p>
		<p>
			<label for="<?php echo esc_html( $this->get_field_id( 'idblg_excerpt_more' ) ); ?>"><?php esc_html_e( 'Signs after excerpt', 'idblog-core' ); ?></label>
			<input class="widefat" id="<?php echo esc_html( $this->get_field_id( 'idblg_excerpt_more' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'idblg_excerpt_more' ) ); ?>" type="text" value="<?php echo esc_attr( $idblg_excerpt_more ); ?>" />
		</p>
		<p>
			<label for="<?php echo esc_html( $this->get_field_id( 'idblg_title_length' ) ); ?>"><?php esc_html_e( 'Maximum length of title', 'idblog-core' ); ?></label>
			<input class="widefat" id="<?php echo esc_html( $this->get_field_id( 'idblg_title_length' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'idblg_title_length' ) ); ?>" type="number" value="<?php echo esc_attr( $idblg_title_length ); ?>" />
		</p>
		<p>
			<input class="checkbox" type="checkbox" <?php checked( $idblg_hide_current_post ); ?> id="<?php echo esc_html( $this->get_field_id( 'idblg_hide_current_post' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'idblg_hide_current_post' ) ); ?>" />
			<label for="<?php echo esc_html( $this->get_field_id( 'idblg_hide_current_post' ) ); ?>"><?php esc_html_e( 'Do not list the current post?', 'idblog-core' ); ?></label>
		</p>
		<p>
			<input class="checkbox" type="checkbox" <?php checked( $idblg_show_author ); ?> id="<?php echo esc_html( $this->get_field_id( 'idblg_show_author' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'idblg_show_author' ) ); ?>" />
			<label for="<?php echo esc_html( $this->get_field_id( 'idblg_show_author' ) ); ?>"><?php esc_html_e( 'Show author?', 'idblog-core' ); ?></label>
		</p>
		<p>
			<input class="checkbox" type="checkbox" <?php checked( $idblg_show_categories ); ?> id="<?php echo esc_html( $this->get_field_id( 'idblg_show_categories' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'idblg_show_categories' ) ); ?>" />
			<label for="<?php echo esc_html( $this->get_field_id( 'idblg_show_categories' ) ); ?>"><?php esc_html_e( 'Show Categories?', 'idblog-core' ); ?></label>
		</p>
		<p>
			<input class="checkbox" type="checkbox" <?php checked( $idblg_show_comments_number ); ?> id="<?php echo esc_html( $this->get_field_id( 'idblg_show_comments_number' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'idblg_show_comments_number' ) ); ?>" />
			<label for="<?php echo esc_html( $this->get_field_id( 'idblg_show_comments_number' ) ); ?>"><?php esc_html_e( 'Show Comments Number?', 'idblog-core' ); ?></label>
		</p>
		<p>
			<input class="checkbox" type="checkbox" <?php checked( $idblg_show_view ); ?> id="<?php echo esc_html( $this->get_field_id( 'idblg_show_view' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'idblg_show_view' ) ); ?>" />
			<label for="<?php echo esc_html( $this->get_field_id( 'idblg_show_view' ) ); ?>"><?php esc_html_e( 'Show View Count?', 'idblog-core' ); ?></label>
		</p>
		<p>
			<input class="checkbox" type="checkbox" <?php checked( $idblg_show_excerpt ); ?> id="<?php echo esc_html( $this->get_field_id( 'idblg_show_excerpt' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'idblg_show_excerpt' ) ); ?>" />
			<label for="<?php echo esc_html( $this->get_field_id( 'idblg_show_excerpt' ) ); ?>"><?php esc_html_e( 'Show Excerpt?', 'idblog-core' ); ?></label>
		</p>
		<p>
			<input class="checkbox" type="checkbox" <?php checked( $idblg_show_thumb ); ?> id="<?php echo esc_html( $this->get_field_id( 'idblg_show_thumb' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'idblg_show_thumb' ) ); ?>" />
			<label for="<?php echo esc_html( $this->get_field_id( 'idblg_show_thumb' ) ); ?>"><?php esc_html_e( 'Show Thumbnail?', 'idblog-core' ); ?></label>
		</p>
		<p>
			<label for="<?php echo esc_html( $this->get_field_id( 'bgcolor' ) ); ?>"><?php esc_html_e( 'Background Color', 'idblog-core' ); ?></label><br />
			<input class="widefat color-picker" id="<?php echo esc_html( $this->get_field_id( 'bgcolor' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'bgcolor' ) ); ?>" type="text" value="<?php echo esc_attr( $bgcolor ); ?>" data-default-color="" />
		</p>
		<p>
			<label for="<?php echo esc_html( $this->get_field_id( 'color_text' ) ); ?>"><?php esc_html_e( 'Text Color', 'idblog-core' ); ?></label><br />
			<input class="widefat color-picker" id="<?php echo esc_html( $this->get_field_id( 'color_text' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'color_text' ) ); ?>" type="text" value="<?php echo esc_attr( $color_text ); ?>" data-default-color="" />
		</p>
		<p>
			<label for="<?php echo esc_html( $this->get_field_id( 'color_link' ) ); ?>"><?php esc_html_e( 'Link Color', 'idblog-core' ); ?></label><br />
			<input class="widefat color-picker" id="<?php echo esc_html( $this->get_field_id( 'color_link' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'color_link' ) ); ?>" type="text" value="<?php echo esc_attr( $color_link ); ?>" data-default-color="" />
		</p>
		<p>
			<label for="<?php echo esc_html( $this->get_field_id( 'color_meta' ) ); ?>"><?php esc_html_e( 'Meta Color', 'idblog-core' ); ?></label><br />
			<input class="widefat color-picker" id="<?php echo esc_html( $this->get_field_id( 'color_meta' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'color_meta' ) ); ?>" type="text" value="<?php echo esc_attr( $color_meta ); ?>" data-default-color="" />
		</p>
		<p>
			<label for="<?php echo esc_html( $this->get_field_id( 'idblg_bannerinfeed' ) ); ?>"><?php esc_html_e( 'Banner After First Post:', 'idblog-core' ); ?></label>
			<textarea class="widefat" rows="6" id="<?php echo esc_html( $this->get_field_id( 'idblg_bannerinfeed' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'idblg_bannerinfeed' ) ); ?>"><?php echo esc_textarea( $instance['idblg_bannerinfeed'] ); ?></textarea>
		</p>
		<?php
	}

	/**
	 * Return the array index of a given ID
	 *
	 * @param Array  $arr Arr.
	 * @param Number $id Post ID.
	 *
	 * @since 1.0.0
	 * @access private
	 */
	private function get_cat_parent_index( $arr, $id ) {
		$len = count( $arr );
		if ( 0 === $len ) {
			return false;
		}
		$id = absint( $id );
		for ( $i = 0; $i < $len; $i++ ) {
			if ( $id === $arr[ $i ]['id'] ) {
				return $i;
			}
		}
		return false;
	}

	/**
	 * Returns the assigned categories of a post in a string
	 *
	 * @param Number $id Post ID.
	 * @access   private
	 * @since     1.0.0
	 */
	private function get_the_categories( $id ) {
		$terms = get_the_terms( $id, 'category' );

		if ( is_wp_error( $terms ) ) {
			return __( 'Error on listing categories', 'idblog-core' );
		}

		if ( empty( $terms ) ) {
			$text = __( 'No categories', 'idblog-core' );
			return $text;
		}

		$categories = array();

		foreach ( $terms as $term ) {
			$categories[] = $term->name;
		}

		$string  = __( 'In', 'idblog-core' ) . ' ';
		$string .= join( ', ', $categories );

		return $string;
	}

	/**
	 * Returns the shortened excerpt, must use in a loop.
	 *
	 * @since 1.0.0
	 * @param Number $excerpt_length Excerpt text to show.
	 * @param String $excerpt_more Text.
	 * @access private
	 */
	private function get_the_trimmed_excerpt( $excerpt_length = 55, $excerpt_more = ' [&hellip;]' ) {

		$post = get_post();

		if ( empty( $post ) ) {
			return '';
		}

		if ( post_password_required( $post ) ) {
			return esc_html__( 'There is no excerpt because this is a protected post.', 'idblog-core' );
		}

		// get excerpt from text field.
		$excerpt = $post->post_excerpt;

		// text processings if no manual excerpt is available.
		if ( empty( $excerpt ) ) {

			// get excerpt from post content.
			$excerpt = strip_shortcodes( get_the_content( '' ) );
			$excerpt = apply_filters( 'the_content', $excerpt );
			$excerpt = str_replace( ']]>', ']]&gt;', $excerpt );
			$excerpt = wp_trim_words( $excerpt, $excerpt_length, $excerpt_more );

			// if excerpt is longer than desired.
			if ( mb_strlen( $excerpt ) > $excerpt_length ) {
				// get excerpt in desired length.
				$sub_excerpt = mb_substr( $excerpt, 0, $excerpt_length );
				// get array of shortened excerpt words.
				$excerpt_words = explode( ' ', $sub_excerpt );
				// get the length of the last word in the shortened excerpt.
				$excerpt_cut = - ( mb_strlen( $excerpt_words[ count( $excerpt_words ) - 1 ] ) );
				// if there is no empty string.
				if ( $excerpt_cut < 0 ) {
					// get the shorter excerpt until the last word.
					$excerpt = mb_substr( $sub_excerpt, 0, $excerpt_cut );
				} else {
					// get the shortened excerpt.
					$excerpt = $sub_excerpt;
				}
			}

			// append 'more' text.
			$excerpt .= $excerpt_more;

		}

		// return text.
		return $excerpt;
	}

	/**
	 * Returns the shortened post title, must use in a loop.
	 *
	 * @param Number $len Number text to show.
	 * @param String $more Text.
	 * @since 1.0.0
	 */
	private function get_the_trimmed_post_title( $len = 40, $more = '&hellip;' ) {

		// get current post's post_title.
		$post_title = get_the_title();

		// if post_title is longer than desired.
		if ( mb_strlen( $post_title ) > $len ) {
			// get post_title in desired length.
			$post_title = mb_substr( $post_title, 0, $len );
			// append ellipses.
			$post_title .= $more;
		}
		// return text.
		return $post_title;
	}

}

add_action(
	'widgets_init',
	function() {
		register_widget( 'Idblog_Mostview_Wgt' );
	}
);
